<?php
/**
 * Plugin Name: Silver Light Galactday Marker
 * Plugin URI:  https://robertsachristan.com
 * Description: Displays the Galactday above the content of each post.
 * Version:     1.0
 * Author:      Robert Sachristan
 * Author URI:  https://robertsachristan.com
 * License:     GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: silver-light-galactday-marker
 */

// Ensure WordPress is loaded before running plugin code
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Converts a given Earth date string into a Galactday format.
 * This implementation uses January 1, 1970 (Unix Epoch) as Galactday 00000.0
 * and calculates the galactday based on 1000 galactday units per Earth year (365.25 days).
 *
 * @param string $dateString The date in 'YYYY-MM-DD HH:MM:SS' format (from post_date).
 * @return string The calculated galactday, formatted to one decimal place, or "Invalid Date".
 */
function slgm_convert_to_galactday( $dateString ) {
    if ( empty( $dateString ) ) {
        return "Invalid Date";
    }

    // Create DateTime objects for the input date and the reference date (Unix Epoch)
    try {
        $inputDate = new DateTime( $dateString, new DateTimeZone('UTC') ); // Treat input as UTC
        $referenceDate = new DateTime( '1970-01-01 00:00:00', new DateTimeZone('UTC') );
    } catch ( Exception $e ) {
        // Log the error if date parsing fails
        error_log( 'Galactday Marker Error: ' . $e->getMessage() );
        return "Invalid Date";
    }

    // Calculate the difference in seconds
    $timeDifferenceSeconds = $inputDate->getTimestamp() - $referenceDate->getTimestamp();

    // Define seconds in a standard Earth year (accounting for leap years on average)
    // Using define() for global constants, which is more compatible.
    if (!defined('SECONDS_PER_EARTH_YEAR')) {
        define('SECONDS_PER_EARTH_YEAR', 365.25 * 24 * 60 * 60);
    }

    // Galactday units per Earth year (a common fan interpretation)
    if (!defined('GALACTDAY_UNITS_PER_YEAR')) {
        define('GALACTDAY_UNITS_PER_YEAR', 1000);
    }

    // Calculate galactday: (time difference in years) * galactday units per year
    $galactday = ( $timeDifferenceSeconds / SECONDS_PER_EARTH_YEAR ) * GALACTDAY_UNITS_PER_YEAR;

    // Format to one decimal place
    return number_format( $galactday, 1, '.', '' );
}

/**
 * Prepends the calculated galactday to the post content.
 *
 * @param string $content The post content.
 * @return string The modified post content with the galactday prepended.
 */
function slgm_display_galactday_above_content( $content ) {
    // Check if we are on a single post page and not in the admin area
    if ( is_single() && in_the_loop() && ! is_admin() ) {
        // Get the post's publication date
        $post_date = get_the_date( 'Y-m-d H:i:s' ); // Get full date and time string

        // Convert to galactday
        $galactday = slgm_convert_to_galactday( $post_date );

        // Create the galactday display HTML
        $galactday_html = '<p style="font-size: 1.1em; font-weight: bold; color: #58a6ff; margin-bottom: 1em;">' .
                         'Galactday: ' . esc_html( $galactday ) .
                         '</p>';

        // Prepend the galactday HTML to the content
        $content = $galactday_html . $content;
    }

    return $content;
}

// Hook into the 'the_content' filter to modify post content
add_filter( 'the_content', 'slgm_display_galactday_above_content' );
?>
